<?php
namespace TwentyI\WHMCS;
use Illuminate\Database\Capsule\Manager as Capsule;
/**
 * This class is for common WHMCS activities
 */
class Provisioning extends Base {
       /**
     	* @return self|null
     	*/
	public static function any() {
		$server_config = Capsule::table('tblservers')
			->where("type", "=", "provision20i")
			->first();
		$product_config = Capsule::table('tblproducts')
			->where("servertype", "=", "provision20i")
			->first();
		if($server_config and $product_config) {
			$password = Decrypt($server_config->password);
			return new self(
					(object)["password" => $password],
					$product_config
				       );
		} else {
			return null;
		}
	}

	public static $CONFIG_OPTIONS = [];

	/**
	 * Returns the options for configuration of the hosting product.
	 * @return array
	 */
	public static function configOptions($configOptions=null) {
		$options = [
			"typeCode" => [ // 1
			"FriendlyName" => "Type Code",
			"Type" => "text",
			"Size" => 32,
			"Description" => "The type of this hosting service",
			],
			"adminUser" => [ // 2
				"FriendlyName" => "WHMCS Admin user",
			"Type" => "dropdown",
			"Options" => [],
			"Description" => "A WHMCS admin user, required to correctly complete setups",
			],
			"stackUserCustomField" => [ // 3
				"FriendlyName" => "Stack User custom field",
			"Type" => "dropdown",
			"Options" => [],
			"Description" => "The custom field you're using for stack user details, see documentation",
			],
		];
                $options['stackUserCustomField']['Options'] = $configOptions['stackUserCustomField']['Options'];
                $options['adminUser']['Options'] = $configOptions['adminUser']['Options'];
		$server_config = Capsule::table('tblservers')
			->where("type", "=", "provision20i")
			->first();
		if($server_config) {
			$password = Decrypt($server_config->password);
			if($password) {
				list($services_key) = explode("+", $password);
				$services_api = new \TwentyI\API\Services($services_key);
				$services_api->setUserAgent(self::userAgent());
				$types = $services_api->getWithFields(
						"/reseller/*/packageTypes"
						);
				$type_options = [];
				foreach($types as $type=>$settings) {
					if($settings->platform === 'virtual') {
						unset($types[$type]);
					 } else {
					 $type_options[$settings->id] = $settings->label;
					}
				}
				$options["typeCode"]["Type"] = "dropdown";
				$options["typeCode"]["Options"] = $type_options;
			}
		}
		return $options;
	}

    /**
     * Returns the options for configuration of the VPS product.
     * @return array
    */
    public static function vpsConfigOptions($configOptions) {
        $options = [
            "adminUser" => [ // 1
            "FriendlyName" => "WHMCS Admin user",
            "Type" => "dropdown",
            "Options" => [],
            "Description" => "A WHMCS admin user, required to correctly complete setups",
            ],
            "stackUserCustomField" => [ // 2
                "FriendlyName" => "Stack User custom field",
            "Type" => "dropdown",
            "Options" => [],
            "Description" => "The custom field you're using for stack user details, see documentation",
            ],
        ];
        $options['stackUserCustomField']['Options'] = $configOptions['stackUserCustomField']['Options'];
        $options['adminUser']['Options'] = $configOptions['adminUser']['Options'];
        return $options;
    }

	/**
	 * Returns an object given WHMCS params.
	 *
	 * @param array $params {
	 *     @var string $serverpassword
	 * }
	 * @return self
	 */
	public static function fromRequest(array $params) {
		return new self(
				(object) ["password" => $params["serverpassword"]],
				(object) $params,
				$params
			       );
	}

	/**
	 * @property string
	 */
	private $serviceId;

	/**
	 * @property string|null
	 */
	private $storedPackageId;

	/**
	 * Returns the client details part of the request.
	 *
	 * @param array $params {
	 *     @var array $clientsdetails
	 * }
	 * @return array {
	 *     @var string $id
	 *     @var array[] $customfields {
	 *         @var string $id
	 *         @var string $value
	 *     }
	 *     @var string $fullname
	 *     @var string|null $companyname
	 *     @var string $email
	 *     @var string $address1
	 *     @var string $address2
	 *     @var string $city
	 *     @var string $state
	 *     @var string $postcode
	 *     @var string $country ISO3166 alpha2
	 *     @var string $phonenumber
	 *     @var string $phonenumberformatted
	 * }
	 */
	protected function clientsDetails(array $params) {
		return $params["clientsdetails"];
	}

	/**
	 * @property string|null
	 */
	public $domain;

	/**
	 * @property string The type reference
	 */
	public $typeCode;

	/**
         * @property string The VPS type reference
         */
	public $vpsSpec;

	/**
         * @property string The VPS application reference
         */
	public $vpsApplicationId;

	/**
         * @property string The VPS operating system reference
         */
	public $vpsOperatingSystem;

	/**
         * @property string The VPS billing period/frequency reference
         */
	public $vpsBillingPeriod;

    /**
         * @property string The data centre location reference
         */
     public $dcLocation;

	/**
	 * Builds the object given WHMCS params.
	 * @param object $server_config {
	 *     @var string $password
	 * }
	 * @param object $product_config {
	 *     @var mixed $configoption1
	 *     @var mixed $configoption2
	 *     @var mixed $configoption3
	 * }
	 * @param array|null $params {
	 *     @var array $customfields {
	 *         @var string $packageId
	 *     }
	 *     @var string $domain
	 *     @var string $serviceid
	 * }
	 */
	public function __construct($server_config, $product_config, array $params = null) {
		list($services_key, $auth_key) = explode("+", $server_config->password);
		parent::__construct((object) [
				"servicesAPIKey" => $services_key,
				"authAPIKey" => $auth_key,
				"stackUserCustomField" => $product_config->configoption3,
				"adminUser" => $product_config->configoption2,
		], $params);
		$this->typeCode = $product_config->configoption1;
		$this->vpsSpec = $product_config->configoption5;
		$this->vpsApplicationId = $product_config->configoption2;
                $this->vpsOperatingSystem = $product_config->configoption4;
		$this->vpsAdminUser = $product_config->configoption1;
		$this->vpsBillingPeriod = $product_config->configoption6;

		if($params) {
			$this->domain = $params["domain"];
			$this->serviceId = $params["serviceid"];
			$this->storedPackageId = $params["customfields"]["packageId"];
		}
	}

	/**
	 * Property getter
	 *
	 * @param string $n
	 * @return mixed
	 */
	public function __get($n) {
		switch($n) {
			/**
			 * @property string|null $packageId The package ID, if there is one.
			 */
			case "packageId":
				if($this->storedPackageId and $this->storedPackageId == "*") {
					$package_info = $this->servicesAPI->getWithFields(
							"/package/{$this->domain}"
							);
					$this->storedPackageId = $package_info->id;
					if($this->storedPackageId) {
						localAPI(
								"updateClientProduct",
								[
								"serviceid" => $this->serviceId,
								"customfields" => base64_encode(serialize([
										"packageId" => $this->storedPackageId,
								]))
								],
								$this->adminUser
							);
					}
				}
				return $this->storedPackageId;
			default:
				return null;
		}
	}

	/**
	 * Creates the service.
	 *
	 * @param array $params {
	 *     @var array $customfields {
	 *         @var string $otherNames Comma-separated list of other names for
	 *             the site.
	 *     }
	 * }
	 * @throws TwentyI\API\Exception
	 * @return int
	 */
	public function create(array $params) {
		$other_domain_names = $params["customfields"]["otherNames"] ?
			preg_split('/,/', $params["customfields"]["otherNames"]) :
			[];
		$master_stack_user = $this->masterStackUserForRequest($params);
                        if ($params['configoptions']['Data Centre Location'] and $params['configoptions']['Data Centre Location'] == 'UK') {
                                $dcLocation = 'dc_location-uk';
                        } elseif ($params['configoptions']['Data Centre Location'] and $params['configoptions']['Data Centre Location'] == 'USA') {
                                $dcLocation = 'dc_location-usa';
                        }
		$response = $this->servicesAPI->postWithFields(
				"/reseller/*/addWeb",
				[
				"type" => $this->typeCode,
				"domain_name" => $this->domain,
				"extra_domain_names" => $other_domain_names,
				"stackUser" => $master_stack_user,
                "location" => $dcLocation,
				]
				);
		if($response) {
			localAPI(
					"updateClientProduct",
					[
					"serviceid" => $this->serviceId,
					"customfields" => base64_encode(serialize([
							"packageId" => $response->result,
					]))
					],
					$this->adminUser
				);
		}
		return $response;
	}


/**
     * Retrieves the VPS application code ID.
     *
     * @return int The ID of the application the VPS will be provisioned with
     *
     */
    public function vpsApplicationOption($applicationOption) {
        if($applicationOption) {
            $server_config = Capsule::table('tblservers')
                ->where("type", "=", "provision20i")
                ->first();
            if($server_config) {
                $password = Decrypt($server_config->password);
                if($password) {
                    list($services_key) = explode("+", $password);
                    $services_api = new \TwentyI\API\Services($services_key);
                    $services_api->setUserAgent(self::userAgent());
                    $vpsApplications = $services_api->getWithFields(
                            "/platform/vps/Application"
                            );
                    foreach($vpsApplications->result as $application) {
                        if(trim($applicationOption) === trim($application->DisplayName)) {
                            return (int)$application->ApplicationId;
                        }
                    }
                }
            }
        }
    }

    /**
     * Retrieves the VPS operating system ID.
     *
     * @return int The operating system ID
     *
     */
    public function vpsOsOption($osOption) {
        if($osOption) {
            $server_config = Capsule::table('tblservers')
                ->where("type", "=", "provision20i")
                ->first();
            if($server_config) {
                $password = Decrypt($server_config->password);
                if($password) {
                    list($services_key) = explode("+", $password);
                    $services_api = new \TwentyI\API\Services($services_key);
                    $services_api->setUserAgent(self::userAgent());
                    $vpsSystems = $services_api->getWithFields(
                            "/platform/vps/VPSOS"
                            );
                    foreach($vpsSystems->result as $system) {
                        if($osOption === trim($system->DisplayName)) {
                            return $system->LibvirtOsVariant;
                        }
                    }
                }
            }
        }
    }


    /**
     * Assigns the specification for the VPS.
     *
     * @return string The spec which the VPS will be provisioned as
     *
     */
    public function vpsSpecification($osSpec) {
        if($osSpec) {
            $options = [
            "1CPU Cores, 25GB SSD, 1GB RAM" => "vps-a",
            "2CPU Cores, 50GB SSD, 2GB RAM" => "vps-b",
            "4CPU Cores, 75GB SSD, 4GB RAM" => "vps-c",
            "6CPU Cores, 150GB SSD, 8GB RAM" => "vps-d",
            "8CPU Cores, 200GB SSD, 16GB RAM" => "vps-e",
            "10CPU Cores, 400GB SSD, 32GB RAM" => "vps-f",
            "12CPU Cores, 800GB SSD, 64GB RAM" => "vps-g",
            "Managed 1CPU Cores, 25GB SSD, 1GB RAM" => "web-managedvps1set1",
            "Managed 2CPU Cores, 50GB SSD, 2GB RAM" => "web-managedvps2setINF",
            "Managed 4CPU Cores, 75GB SSD, 4GB RAM" => "web-managedvps4setINF",
            "Managed 6CPU Cores, 100GB SSD, 8GB RAM" => "web-managedvps8setINF",
            "Managed 8CPU Cores, 200GB SSD, 16GB RAM" => "web-managedvps16setINF",
            "Managed 10CPU Cores, 400GB SSD, 32GB RAM" => "web-managedvps32setINF",
            "Managed 12CPU Cores, 800GB SSD, 64GB RAM" => "web-managedvps64setINF"
            ];
            foreach($options as $specKey => $spec) {
                if(trim($osSpec) === $specKey) {
                    return $spec;
                }
            }
        }
    }

    /**
     * Creates the VPS service.
     *
     * @param array $params {
     *     @var array $customfields {
     *     }
     * }
     * @throws TwentyI\API\Exception
     * @return int
     */

    public function vpsCreate(array $params) {
        if(!empty($params['customfields']['vpsId'])) {
            throw new \Exception("Cannot create, vpsId has been set/VPS already exists.");
        }
        if(@$params['configoptions']['Operating System']) {
            $vpsOperatingSystem = \TwentyI\WHMCS\Provisioning::vpsOsOption($params['configoptions']['Operating System']);
        }
        if(@$params['configoptions']['Application']) {
            $vpsApplication = \TwentyI\WHMCS\Provisioning::vpsApplicationOption($params['configoptions']['Application']);
        }
        if(empty($params['configoptions']['Application']) || $params['configoptions']['Application'] === "None") {
            unset($vpsApplication);
        }
        if($params['configoptions']['Specification']) {
            $vpsSpecification = \TwentyI\WHMCS\Provisioning::vpsSpecification($params['configoptions']['Specification']);
            $vpsType = "unmanaged";
            if (preg_match("/web/i", $vpsSpecification)) {
                $vpsType = "managed";
            }
        }
        if(empty($params['configoptions']['Specification'])) {
            throw new \Exception("No VPS specification detected!");
        }
        if($params) {
            $order_config = Capsule::table('tblhosting')
                ->where("id", "=", $params['serviceid'])
                ->first();
            $billingCycle = $order_config->billingcycle;
        }
        if($params['configoptions']['Data Centre Location'] and $params['configoptions']['Data Centre Location'] == 'UK') {
            $dcLocation = 'dc_location-uk';
        } elseif ($params['configoptions']['Data Centre Location'] and $params['configoptions']['Data Centre Location'] == 'USA') {
            $dcLocation = 'dc_location-usa';
        }
        if($params['configoptions']['Operating System'] === "Windows Server 2012 R2" || $params['configoptions']['Operating System'] === "Windows Server 2016" && !empty($vpsApplication)){
            throw new \Exception("Cannot provision a Windows virtual machine with an application. Remove the application from the product; or amend the operating system.");
        }
        if($params['configoptions']['Operating System'] === "Windows Server 2012 R2" || $params['configoptions']['Operating System'] === "Windows Server 2016" && $vpsSpecification === "vps-a"){
            throw new \Exception("Cannot provision a Windows virtual machine with the current specification. Select a 2GB specification or more.");
        }
        if($vpsType == "managed") {
            if (!empty($vpsOperatingSystem) || !empty($vpsApplication)) {
                throw new \Exception("Cannot provision a Managed VPS with an operating system, or application set. Adjust your settings via Setup -> Products/Services.");
            }
        } else {
            if(empty($vpsOperatingSystem)) {
                throw new \Exception("No operating system configuration detected!");
            }
        }
        $master_stack_user = $this->masterStackUserForRequest($params);
        switch($billingCycle) {
            case "Monthly":
                $vpsBillingPeriod = "1";
                break;
            case "Annually":
                $vpsBillingPeriod = "12";
                break;
            default:
                throw new \Exception("Product billing cycle must be monthly or yearly.");
        }
        $response = $this->servicesAPI->postWithFields(
                "/reseller/*/addVPS",
                [
                "configuration" => [
                    "Name" => null, /* eg. "vps-1234" */
                    "ApplicationId" => $vpsApplication,
                    "location" => $dcLocation,
                ],
                "forUser" => $master_stack_user, /* eg. "stack-user:588" */
                "options" => [
                    "os" => $vpsOperatingSystem, /* eg. "rhel6". Some of these affect the price. */
                ],
                "periodMonths" => $vpsBillingPeriod, /* The period the service should be purchased for, eg. one month/one year.
                                    "token" => null, /* Required where the id is "auto". */
                "type" => $vpsSpecification, /* eg. "vps-e" */
                ]
                );
        sleep(5);
        if($response) {
            $server_config = Capsule::table('tblservers')
                ->where("type", "=", "provision20i")
                ->first();
            $vps_info = [];
            if($server_config) {
                $password = Decrypt($server_config->password);
                if($password) {
                    list($services_key) = explode("+", $password);
                    $services_api = new \TwentyI\API\Services($services_key);
                    $services_api->setUserAgent(self::userAgent());
                    if(preg_match("/[\s]*(web-)/i", $this->vpsSpec)) {
                        $vps_info = $services_api->getWithFields(
                                "/managed_vps/$response->result"
                                );
                    } else {
                        $vps_info = $services_api->getWithFields(
                                "/vps/$response->result"
                                );
                    }
                }
            }
            $command = 'UpdateClientProduct';
            $postData = array(
                    'serviceid' => $this->serviceId,
                    'domain' => "$vps_info->GuestHostname",
                    'unset' => array("serviceusername", "servicepassword"),
                    'customfields' => base64_encode(serialize(array("vpsId"=>$response->result))),
                    );
            $results = localAPI($command, $postData, $this->vpsAdminUser);
        }
        return $response;
    }

	/**
	 * Removes the service (VPS).
	 *
	 * @throws TwentyI\API\Exception
	 * @return int 1 if removed successfully
	 */
	public function delete($params) {
		if($params["producttype"] === "other") {
			if(preg_match("/[\s]*(vps-)/i", $params['configoption5'])) {
				$contracts = $this->servicesAPI->getWithFields("/contract");
                        	foreach($contracts as $contract) {
                                	if($contract->serviceType === "vps") {
                                        	if($contract->serviceId === $params['customfields']['vpsId']){
                                                $contractID = $contract->id;
                                        	}
                                	}
                        	}
			} elseif(preg_match("/[\s]*(web-)/i", $params['configoption5'])) {
				$contracts = $this->servicesAPI->getWithFields("/contract");
                                foreach($contracts as $contract) {
                                        if($contract->serviceType === "managed_vps") {
                                                if($contract->serviceId === $params['customfields']['vpsId']){
                                                $contractID = $contract->id;
                                                }
                                        }
                                }
			}
			if(!$contractID) {
				throw new \Exception("Cannot find contract ID.");
			}
			return $this->servicesAPI->postWithFields(
					"/contract/$contractID",
					[
					"id" => null,
                			"jsonrpc" => "2.0",
                			"method" => "cancel",
                			"params" => [
                        			"0" => [
                                			"reason" => "VPS cancellation issued via WHMCS.",
                                			]
                        			]
                			]
					);
		} else {
		if(!$this->packageId) {
			throw new \Exception("Cannot delete without package ID");
		}
		return $this->servicesAPI->postWithFields(
				"/reseller/*/updatePackage",
				[
				"id" => [$this->packageId],
				"delete-id" => [$this->packageId],
				]
				);
		}
	}

	/**
	 * Disables the service
	 *
	 * @throws TwentyI\API\Exception
	 * @return mixed
	 */
	public function disable() {
		if(!$this->packageId) {
			throw new \Exception("Cannot disable without package ID");
		}
		return $this->servicesAPI->postWithFields(
				"/package/{$this->packageId}/userStatus",
				[
				"subservices" => [
				"default" => false,
				],
				]
				);
	}

	/**
	 * Enables the service
	 *
	 * @throws TwentyI\API\Exception
	 * @return mixed
	 */
	public function enable() {
		if(!$this->packageId) {
			throw new \Exception("Cannot enable without package ID");
		}
		return $this->servicesAPI->postWithFields(
				"/package/{$this->packageId}/userStatus",
				[
				"subservices" => [
				"default" => true,
				],
				]
				);
	}


	/**
	 * Modifies the type of an existing service
	 *
	 * @param array $params {
	 * }
	 * @throws TwentyI\API\Exception
	 * @return int
	 */
	public function modifyType(array $params) {
		$master_stack_user = $this->masterStackUserForRequest($params);
		$response = $this->servicesAPI->postWithFields(
				"/reseller/*/updatePackage",
				[
				"id" => [
				$this->packageId,
				],
				"packageBundleTypes" => [
				$this->packageId => $this->typeCode,
				]
				]
				);
		return $response->result;
	}

	/*
	 * Renews the existing service (VPS)
         *
         * @param array $params {
         * }
         * @throws TwentyI\API\Exception
         * @return mixed
	*/
    public function renew(array $params) {
        if($params) {
            $order_config = Capsule::table('tblhosting')
                ->where("id", "=", $params['serviceid'])
                ->first();
            $billingCycle = $order_config->billingcycle;
        }
        if($params["producttype"] === "other") {
            $contracts = $this->servicesAPI->getWithFields("/contract");
            foreach($contracts as $con) {
                if(in_array($con->serviceType, ["managed_vps", "vps"]) && $con->serviceId == $params['customfields']['vpsId']){
                    $contract = $con;
                    $contractID = $contract->id;
                }
            }
            if (!$contractID) {
                throw new \Exception("Cannot find contract ID.");
            }
            if (preg_match("/vps/", $contract->serviceType) ) {
                switch($billingCycle) {
                    case "Monthly":
                        $vpsBillingPeriod = "1";
                        break;
                    case "Annually":
                        $vpsBillingPeriod = "12";
                        break;
                    default:
                        throw new \Exception("Product billing cycle must be monthly or yearly.");
                }
            }
            $response = $this->servicesAPI->postWithFields(
                    "/reseller/*/contract/$contractID",
                    [
                    "id" => null,
                    "jsonrpc" => "2.0",
                    "method" => "renew",
                    "params" => [ (int) $vpsBillingPeriod ]
                    ]
                    );
            if($response){
                return "success";
            } else {
                error_log("Got unexpected response: ".var_export($response,true));
                throw new \Exception("Renew command failed.");
            }
        }
    }

    /**
     * Logs into Stack as the service.
     *
     * If the service ref is currently "*", in addition to the ref being
     * autodetected, any existing master stack user will be granted access to
     * the service.
     *
	 * @param array $params
	 * @throws TwentyI\API\Exception
	 * @throws Exception
	 * @return string The single-sign-on URL.
	 */
	public function singleSignOn(array $params) {
		$master_stack_user = $this->existingMasterStackUserForRequest($params);
		if(
				$this->storedPackageId and
				$this->storedPackageId == "*" and
				$master_stack_user
		  ) {
			logModuleCall(
					"provision20i",
					__FUNCTION__,
					$params,
					null,
					"(" . \TwentyI\WHMCS\Base::shortVersion() . ") Overriding access to {$this->domain} to include {$master_stack_user} in StackCP. Please note that this only covers PackageId=*."
				     );
			$this->servicesAPI->postWithFields(
					"/reseller/*/susers",
					[
					"grant_map" => [
					$master_stack_user => [
					$this->packageId => true,
					]
					]
					]
					);
		}
		if($master_stack_user) {
			$token_info =
				$this->authAPI->controlPanelTokenForUser($master_stack_user);
			if($this->packageId) {
				$package_info = $this->servicesAPI->getWithFields(
						"/package/{$this->packageId}"
						);
				return $this->servicesAPI->singleSignOn(
						$token_info->access_token,
						$package_info->name
						);
			} else {
				return $this->servicesAPI->singleSignOn(
						$token_info->access_token,
						null
						);
			}
		} else {
			if(!$this->packageId) {
				throw new \Exception("Cannot single-sign-on without package ID");
			}
			$package_info = $this->servicesAPI->getWithFields(
					"/package/{$this->packageId}"
					);
			if(!$package_info->stackUsers[0]) {
				throw new \Exception("No stack users to authenticate");
			}
			$token_info =
				$this->authAPI->controlPanelTokenForUser(
						$package_info->stackUsers[0]
						);
			return $this->servicesAPI->singleSignOn(
					$token_info->access_token,
					$package_info->name
					);
		}
	}

	public function usageUpdate(array $params)
	{
		$server_password = $params['serverpassword'];
		// Run connection to retrieve usage for all domains/accounts on $serverid
		if ($server_password) {
			list($services_key) = explode("+", $server_password);
			$services_api = new \TwentyI\API\Services($services_key);
			$packages = $services_api->getWithFields(
				"/package"
			);
			$results = [];
			foreach ($packages as $package) {
				$id = $package->id;
				$domain = $package->name;
				$packageUsage = $services_api->getWithFields(
					"/package/$id/web/usage"
				);
				$packageLimits = $services_api->getWithFields(
					"/package/$id/limits"
				);
				if ($packageUsage && $packageLimits) {
					$results[$domain] = $packageUsage;
					$results[$domain]->bandwidthLimit = $packageLimits->bandwidth;
					$results[$domain]->diskLimit = $packageLimits->webspace;
				} else {
					//trigger_error("Skipping usage update for $domain as no usage or limits found");
				}
			}
		}
		return $results;
	}
}
