if (typeof jQuery === 'undefined') {
    throw new Error("bootstrap-imageupload's JavaScript requires jQuery.");
}

(function($) {
    'use strict';

    var options = {};

    var methods = {
        init: init,
        disable: disable,
        enable: enable,
        reset: reset
    };

    // Plugin Definition
    $.fn.imageupload = function(methodOrOptions) {
        var givenArguments = arguments;

        return this.filter('div').each(function() {
            if (methods[methodOrOptions]) {
                methods[methodOrOptions].apply($(this), Array.prototype.slice.call(givenArguments, 1));
            }
            else if (typeof methodOrOptions === 'object' || !methodOrOptions) {
                methods.init.apply($(this), givenArguments);
            }
            else {
                throw new Error('Method "' + methodOrOptions + '" is not defined for imageupload.');
            }
        });
    };

    $.fn.imageupload.defaultOptions = {
        allowedFormats: ['jpg', 'jpeg', 'png', 'gif', 'mp4'],
        maxWidth: 250,
        maxHeight: 250,
        maxFileSizeKb: 2048
    };

    function init(givenOptions) {
        options = $.extend({}, $.fn.imageupload.defaultOptions, givenOptions);

        var $imageupload = this;
        var $fileTab = $imageupload.find('.file-tab');
        var $fileTabButton = $imageupload.find('.panel-heading .btn:eq(0)');
        var $browseFileButton = $fileTab.find('input[type="file"]');
        var $removeFileButton = $fileTab.find('.btn:eq(1)');
        var $uploadButton = $('#uploadimg'); // The ID of the upload button

        resetFileTab($fileTab);
        showFileTab($fileTab);
        enable.call($imageupload);

        $fileTabButton.off();
        $browseFileButton.off();
        $removeFileButton.off();

        $uploadButton.hide();

        $fileTabButton.on('click', function() {
            $(this).blur();
            showFileTab($fileTab);
        });

        $browseFileButton.on('change', function() {
            $(this).blur();
            submitImageFile($fileTab).then(() => {
                $uploadButton.show();
            });
        });

        $removeFileButton.on('click', function() {
            $(this).blur();
            resetFileTab($fileTab);
            $uploadButton.hide();
        });
    }

    function disable() {
        var $imageupload = this;
        $imageupload.addClass('imageupload-disabled');
    }

    function enable() {
        var $imageupload = this;
        $imageupload.removeClass('imageupload-disabled');
    }

    function reset() {
        var $imageupload = this;
        init.call($imageupload, options);
    }

    function getAlertHtml(message) {
        var html = [];
        html.push('<div class="alert alert-danger alert-dismissible">');
        html.push('<button type="button" class="close" data-dismiss="alert">');
        html.push('<span>&times;</span>');
        html.push('</button>' + message);
        html.push('</div>');
        return html.join('');
    }

    function getImageThumbnailHtml(src) {
        return '<img src="' + src + '" alt="Image preview" class="thumbnail" style="max-width: ' + options.maxWidth + 'px; max-height: ' + options.maxHeight + 'px">';
    }

    function getVideoThumbnailHtml(src) {
        return `<video playsinline muted class="thumbnail" style="max-width: ${options.maxWidth}px; max-height: ${options.maxHeight}px;">
                    <source src="${src}" type="video/mp4">
                    Your browser does not support the video tag.
                </video>`;
    }

    function getFileExtension(path) {
        return path.substr(path.lastIndexOf('.') + 1).toLowerCase();
    }

    function isValidImageFile(file) {
        return new Promise((resolve, reject) => {
            if (file.size / 1024 > options.maxFileSizeKb) {
                reject('File is too large (max ' + options.maxFileSizeKb + 'kB).');
                return;
            }

            var fileExtension = getFileExtension(file.name);
            if ($.inArray(fileExtension, options.allowedFormats) > -1) {
                resolve('File is valid.');
            } else {
                reject('File type is not allowed.');
            }
        });
    }

    function showFileTab($fileTab) {
        var $imageupload = $fileTab.closest('.imageupload');
        var $fileTabButton = $imageupload.find('.panel-heading .btn:eq(0)');

        if (!$fileTabButton.hasClass('active')) {
            $imageupload.find('.panel-heading .btn:eq(1)').removeClass('active');
            $fileTabButton.addClass('active');
            $fileTab.show();
            resetFileTab($fileTab);
        }
    }

    function resetFileTab($fileTab) {
        $fileTab.find('.alert').remove();
        $fileTab.find('img, video').remove();
        $fileTab.find('.btn span').text('Browse');
        $fileTab.find('.btn:eq(1)').hide();
        $fileTab.find('input').val('');
    }

    function submitImageFile($fileTab) {
        return new Promise((resolve, reject) => {
            var $browseFileButton = $fileTab.find('.btn:eq(0)');
            var $removeFileButton = $fileTab.find('.btn:eq(1)');
            var $fileInput = $browseFileButton.find('input');

            $fileTab.find('.alert').remove();
            $fileTab.find('img, video').remove();
            $browseFileButton.find('span').text('Browse');
            $removeFileButton.hide();

            if (!($fileInput[0].files && $fileInput[0].files[0])) {
                resolve();
                return;
            }

            $fileTab.prepend('<div class="thumbnail loading-spinner"><i class="fad fa-circle-notch fa-spin fa-2x"></i></div>');  // Show loading spinner
            $browseFileButton.prop('disabled', true);
            var file = $fileInput[0].files[0];

            isValidImageFile(file)
                .then(message => {
                    var fileReader = new FileReader();

                    fileReader.onload = function(e) {
                        $fileTab.find('.loading-spinner').remove();  // Hide loading spinner

                        var fileExtension = getFileExtension(file.name);
                        if (fileExtension === 'mp4') {
                            $fileTab.prepend(getVideoThumbnailHtml(e.target.result));
                        } else {
                            $fileTab.prepend(getImageThumbnailHtml(e.target.result));
                        }

                        $browseFileButton.find('span').text('Change');
                        $removeFileButton.css('display', 'inline-block');
                        resolve();
                    };

                    fileReader.onerror = function() {
                        $fileTab.find('.loading-spinner').remove();  // Hide loading spinner
                        $fileTab.prepend(getAlertHtml('Error loading file.'));
                        $fileInput.val('');
                        reject('Error loading file.');
                    };

                    fileReader.readAsDataURL(file);
                })
                .catch(error => {
                    $fileTab.find('.loading-spinner').remove();  // Hide loading spinner
                    $fileTab.prepend(getAlertHtml(error));
                    $browseFileButton.find('span').text('Browse');
                    $fileInput.val('');
                    reject(error);
                })
                .finally(() => {
                    $browseFileButton.prop('disabled', false);
                });
        });
    }

}(jQuery));
